<?php

namespace App\Console\Commands;

use App\Models\ResellerSmsGetway;
use Illuminate\Console\Command;

class UpdateResellerSmsBodyCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'reseller:sms-body-update';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update SMS body from default_sms_array for all existing ResellerSmsGetway records';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Starting SMS body update process...');

        // Define default SMS types with their default values
        $default_sms_array = array(
            [
                'sms_type' => 'other_reseller_customer_recharge',
                'send_sms' => 'No',
                'sms_body' => 'Dear {c_name}. CID: {c_id}. Package: {BTRC_speed}MB. Month: {month}. MRC: {BTRC_rate}. Thanks for the payment.',
            ],[
                'sms_type' => 'other_reseller_customer_reminder',
                'send_sms' => 'No',
                'sms_body' => 'সম্মানিত গ্রাহক, আইডি  {c_id}. ইন্টারনেট বিল {BTRC_rate} পরিশোধের শেষ তারিখ {c_expire_date} দয়া করে আপনার বিলটি পরিশোধ করুন।',
                'days_before' => '2',
            ],[
                'sms_type' => 'other_reseller_customer_expire',
                'send_sms' => 'No',
                'sms_body' => 'প্রিয় গ্রাহক {c_name} আপনার ইন্টারনেট বিল পরিশোধ করার সময়সীমা অতিক্রম করেছে। নিরবিচ্ছিন্ন সংযোগ পেতে বকেয়া টাকা পরিশোধ করুন।',
            ],[
                'sms_type' => 'other_reseller_customer_create',
                'send_sms' => 'No',
                'sms_body' => '{Delta Software and Communication Ltd} এ আপনাকে স্বাগতম। আপনার {c_id} সংরক্ষন করে রাখুন।',
            ],[
                'sms_type' => 'own_reseller_customer_create',
                'send_sms' => 'No',
                'sms_body' => '{Delta Software and Communication Ltd} এ আপনাকে স্বাগতম। আপনার {c_id} সংরক্ষন করে রাখুন।',
            ],[
                'sms_type' => 'own_reseller_customer_payment',
                'send_sms' => 'No',
                'sms_body' => 'Dear {c_name}. CID: {c_id}. Package: {BTRC_speed}MB. Month: {payment_description}. MRC: {PaymentAmount}. Thanks for the payment.',
            ],[
                'sms_type' => 'own_reseller_customer_reminder',
                'send_sms' => 'No',
                'sms_body' => 'সম্মানিত গ্রাহক, আইডি {c_id}. ইন্টারনেট বিল {BTRC_rate} পরিশোধের শেষ তারিখ {c_deadline} দয়া করে আপনার বিলটি পরিশোধ করুন।',
                'days_before' => '2',
            ],[
                'sms_type' => 'own_reseller_customer_expire',
                'send_sms' => 'No',
                'sms_body' => 'প্রিয় গ্রাহক {c_name} আপনার ইন্টারনেট বিল পরিশোধ করার সময়সীমা অতিক্রম করেছে। নিরবিচ্ছিন্ন সংযোগ পেতে বকেয়া টাকা পরিশোধ করুন।',
            ]
        );

        // Create a map of default SMS types for easy lookup
        $default_sms_map = [];
        foreach ($default_sms_array as $default_sms) {
            $default_sms_map[$default_sms['sms_type']] = $default_sms;
        }

        // Get all ResellerSmsGetway records
        $sms_getways = ResellerSmsGetway::all();
        $total_records = $sms_getways->count();
        $updated_count = 0;
        $added_count = 0;

        $this->info("Found {$total_records} ResellerSmsGetway record(s) to process.");

        $bar = $this->output->createProgressBar($total_records);
        $bar->start();

        foreach ($sms_getways as $sms_getway) {
            $existing_sms_array = json_decode($sms_getway->sms_formate, true);
            $existing_types = array_column($existing_sms_array, 'sms_type');
            $needs_update = false;
            $record_updated = false;
            $record_added = false;

            // Update sms_body for existing SMS types from default_sms_array
            foreach ($existing_sms_array as &$existing_sms) {
                if (isset($default_sms_map[$existing_sms['sms_type']])) {
                    $default_sms = $default_sms_map[$existing_sms['sms_type']];
                    if (isset($default_sms['sms_body'])) {
                        $existing_sms['sms_body'] = $default_sms['sms_body'];
                        $needs_update = true;
                        $record_updated = true;
                    }
                }
            }

            // Check each default type and add if missing
            foreach ($default_sms_array as $default_sms) {
                if (!in_array($default_sms['sms_type'], $existing_types)) {
                    $existing_sms_array[] = $default_sms;
                    $needs_update = true;
                    $record_added = true;
                }
            }

            // Update database if changes were made
            if ($needs_update) {
                $sms_getway->sms_formate = json_encode($existing_sms_array);
                $sms_getway->save();

                if ($record_updated) {
                    $updated_count++;
                }
                if ($record_added) {
                    $added_count++;
                }
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        $this->info("Process completed!");
        $this->info("Total records processed: {$total_records}");
        $this->info("Records with SMS body updated: {$updated_count}");
        $this->info("Records with missing SMS types added: {$added_count}");

        return Command::SUCCESS;
    }
}

